#!/usr/bin/perl -w

# ClonePanel - Manages duplicate accounts on two or more webservers,
# including snapshot backups, monitoring and failover dns.
# Copyright (C)2006 Chris Cheers, Internet Lynx.
# Contact chris[at]clonepanel[dot]com.
# Internet Lynx, PO Box 7117, Mannering Park, NSW 2259, Australia

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Version=0.33
# set_dns.pl - perl script reads a zone file and installs to DNS servers
# Should be called only by set_dns shell script

use strict;
use lib "/usr/local/cpanel";
# Standard file location on a CPanel server - YMMV
# Specifically in jailshell this may not work if that location is not available to you
# Alternatively a copy of the Cpanel directory (or just Cpanel/Accounting.pm)
# can be placed in the included clonepanel directory

use Cpanel::Accounting;
use Net::DNS;
use Net::DNS::ZoneFile::Fast;
use HTML::TreeBuilder;

my ($dir, $authdir, $zonefn, $host, $ip, $method, @domains) = @ARGV;
$dir =~ s/[^\w\.\-\/]//g if $dir;
$authdir =~ s/[^\w\.\-\/]//g if $authdir;

$zonefn =~ s/[^\w\.\-]//g if $zonefn;
$host =~ s/\W//g if $host;
$ip =~ s/[^\d\.]//g if $ip;
$method =~ s/\W//g if $method;
@domains = cleandomainnames(@domains);

die "Usage:  ./set_dns.pl dir authdir zonefilename host ip method[:port] zone1 [zone2 zone3 ...]" unless @domains;


foreach my $domain (@domains) {
	$domain =~ s/#.*//;
	next unless $domain =~ /\w/;
	my $zonefile = "$dir/$domain/$zonefn";
	print "Fetching zone $domain from $zonefile... ";
	my $rzone = readzonefile($domain,$zonefile)
		or die "Failed to read zone $domain from local file";
	print "Done\n";
	print "Setting zone $domain on $host using $method\n";
	my ($output,$test);
	if ($method eq 'WHM') {
		$output=setzonewhm($domain,$host,$ip,$rzone);
		$test='Zone Modified';
	} elsif ($method=~/^DA(\d*)/) {
		$output=setzoneda($domain,$host,$ip,$1,$rzone);
		$test='Raw zone saved';
	} elsif ($method eq 'BIND') {
		$output=setzonebind($domain,$host,$ip,$rzone);
		$test='NOERROR';
	} else {
		die "Unsupported DNS update method: $method";
	}
	if ( $output =~/$test/ ) {			# Success!
		print "$output\nDone\n";
	} else {
		die "Zone update failed:\n$output\n";
	}
}

sub readzonefile {
	my ($domain,$filename) = @_;
	open INFILE, $filename or die "Can't open $filename for read";
	my $zonetext;
	{									#localise this block for slurp
		local $/;
		$zonetext = <INFILE>;
	} 
	close INFILE;
	my $rzone = Net::DNS::ZoneFile::Fast::parse($zonetext) or die "Failed to create zone";
	return $rzone;						#return a listref of RR records
}
	


sub setzonebind {
	my ($domain,$nameserver,$ip,$rzone)=@_;
#	my $auth = getauth($nameserver);
#	die "Unknown " unless $auth->{user};

	my $update = Net::DNS::Update->new("$domain.");		# Create the update packet.
	$update->push(update => rr_del("$domain."));		# Delete all records for the main domain name.
	my $subs = {$domain=>1};							# and keep a note that it's been done
	foreach (@$rzone) {
		my $name = $_->name;
		unless (exists $subs->{$name}) {				# new subdomain - make update delete it
			$subs->{$name}=1;
			$update->push(update => rr_del("$name."));
		}
		$update->push(update => rr_add($_->string));	# then replace with new info
	}
# Now send the update to the specified nameserver (master or stealth master).
	my $res = Net::DNS::Resolver->new;
	$res->nameservers($ip);
	$res->udp_timeout(10);
	my $reply = $res->send($update);
	if ($reply) {						# did it work?
		my $status = $reply->header->rcode eq 'NOERROR' ? 'ok' : 'failed';
		return "Update $status: " . $reply->header->rcode;
	} else {
		return 'Update failed: ' . $res->errorstring;
	}
}

sub setzonewhm {
	my ($domain,$whmserver,$ip,$rzone)=@_;
	my ($whm) = Cpanel::Accounting->new;
	my $auth = getauth($whmserver);
	
	die "Unknown WHM host" unless $auth->{user};
	$whm->{host} = $ip;
	$whm->{user} = $auth->{user};
	$whm->{accesshash} = $auth->{accesshash};
	$whm->{usessl} = 1;
	
	my $page = getzones($whm);
	die "Error - ModAccounting returned [$whm->{error}]" if $whm->{error} ne "";
	my $pagetree = HTML::TreeBuilder->new_from_content($page);
	my $heading;
	my @s = $pagetree->look_down('_tag', 'select');		#all select items on page
	my @opvals;
	foreach my $s (@s) {
		next unless $s->{name} eq 'domainselect';		#find select element containing domains available
		my @options = $s->look_down('_tag', 'option');
		foreach my $o (@options) { push @opvals,$o->{value} };	#create list of domain names
		last
	}
	$pagetree->delete;
	die "No domains found" unless @opvals;
	die "$domain not available for edit" unless grep /^$domain$/,@opvals;

	my $formdata = zone_to_cpform($domain,$rzone);
	
	$page = sendzone($whm,$formdata);
	$pagetree = HTML::TreeBuilder->new_from_content($page);
	my $body = $pagetree->find('body');		#keep only body of page
	my $output = $body->as_HTML;
	$pagetree->delete;

	return $output
}

sub setzoneda {
	require LWP::UserAgent;
	require MIME::Base64;

	my ($domain,$daserver,$ip,$port,$rzone)=@_;
	$port ||= 2222;

	my $auth = getauth($daserver);
	my $user = $auth->{user};
	my $pass = $auth->{accesshash};
	
	die "Unknown DA host" unless $auth->{user};

	my $proto = 'https';		# ssl only
	my $cmd = "CMD_API_DNS_ADMIN?domain=$domain&action=rawsave";

#	my $page = getzones($whm);
#	die "Error - ModAccounting returned [$whm->{error}]" if $whm->{error} ne "";
#	my $pagetree = HTML::TreeBuilder->new_from_content($page);
#	my $heading;
#	my @s = $pagetree->look_down('_tag', 'select');		#all select items on page
#	my @opvals;
#	foreach my $s (@s) {
#		next unless $s->{name} eq 'domainselect';		#find select element containing domains available
#		my @options = $s->look_down('_tag', 'option');
#		foreach my $o (@options) { push @opvals,$o->{value} };	#create list of domain names
#		last
#	}
#	$pagetree->delete;
#	die "No domains found" unless @opvals;
#	die "$domain not available for edit" unless grep /^$domain$/,@opvals;

# TODO - implement similar "domain exists" check for DA API

	my $content = zone_to_da_text($domain,$rzone);

	my $hdrs = new HTTP::Headers(Accept => 'text/plain');
	my $url = "$proto://$ip:$port/$cmd";
	my $req = new HTTP::Request('POST', $url, $hdrs, $content);
	my $ua = new LWP::UserAgent;

# DA doesn't challenge for auth, so standard LWP method doesn't work.
# Instead set up the Basic auth header in advance
	my $auth_header = "Authorization";
	my $auth_value = "Basic " . MIME::Base64::encode("$user:$pass", "");

	$req->header($auth_header => $auth_value);

	my $resp = $ua->request($req);

	return $resp->is_success ? $resp->content : $resp->message;
}

sub zone_to_da_text {		#convert zone record object to DA-compatible text
	my ($domain,$rzone) = @_;
	my $zone = '';
	foreach (@$rzone) {
		$zone .= $_->string . "\n";
	}
# Bind accepts this zone record -> string format but DA doesn't, so change
# the SOA to suit:
	$zone =~ s/$domain.\s+(\d+)\s+IN\s+SOA/\$TTL $1\n\@    IN    SOA/;

	return $zone;
}

sub zone_to_cpform {		#very nasty little routine to convert zone records into WHM-compatible form submission
	my ($domain,$rzone) = @_;

	my $linecount = 100;
	my $formdata = [	"zone=$domain.db",
						"line-0=" . form_encode("; Modified by set_dns.pl via Web Host Manager\n"),
						"line-1=" . form_encode("; Zone File for $domain\n"),
						"line-30=" . form_encode("\$TTL 900"),
						];
	my $sublevel = 0;
	foreach (@$rzone) {
		my $longline = $_->string;
		foreach my $line (split "\n",$longline) {
			unless ($line =~ /\w/) {
				push @$formdata, "line-" . $linecount++ . "=";
				next;												#include blank lines
			}
			my @items;
			my $itemcount = 1;
			if ($line =~ /([^\"]*)\"([^\"]*)\"(.*)/) {
# Line contains a quoted part (likely a TXT record)
				@items = ((split /\s+/,$1),"\"$2\"",(split /\s+/,$3));
			} else {
				@items = split /\s+/,$line;
			}
			foreach my $item (@items) {
				next unless $item =~ /[^\s]/;						#ignore blank items
				last if $item =~ /^;/;								#ignore comments and everything after
				my $indent = $sublevel ? $sublevel : $itemcount++;
				if ($item =~ /\(/) {								#handle fixed indent of ()
					$sublevel = $itemcount - 2;
				} elsif ($item =~ /\)/) {
					$linecount++;									#newline always before )
					$sublevel = 0;									#and reset fixed indent
				}
				push @$formdata, "line-$linecount-$indent=" . form_encode($item);
			}
			$linecount++;
		}
	}

	return $formdata;
}

sub form_encode {
	my $val = shift;
	$val =~ s/([^\w\-\.\@])/sprintf("%%%2.2x",ord($1))/eg;
	return $val;
}

sub getzones {					#substitute for missing routine in Cpanel::Accounting
   my($self) = @_;
   my (@PAGE) = $self->whmreq("/scripts/editdnslist",'GET');
   my $page = join("\n", @PAGE);
   return() if $self->{error} ne "";
   return($page);
}

sub sendzone {					#substitute for missing routine in Cpanel::Accounting
	my($self,$data) = @_;
	my $formdata = join "&", @$data;
	my (@PAGE) = $self->whmreq("/scripts2/savezone","POST",$formdata);
	my $page = join("\n", @PAGE);
	return() if $self->{error} ne "";
	return($page);
}

sub getauth {
	my ($host) = @_;
	unless (	open AUTHFILE, "$authdir/$host.WHM"	# CPanel
		 or	open AUTHFILE, "$authdir/$host.DA"	# DA
			) {
		die "Can't open key file $authdir/$host.WHM or $host.DA";
	}
	my $username = <AUTHFILE>;
	chomp $username;
	my $accesshash;
	{									#localise this block for slurp
		local $/;
		$accesshash = <AUTHFILE>;
	} 
	chomp $accesshash;
	close AUTHFILE;
	return { user => $username, accesshash => $accesshash }
}

sub cleandomainnames {
	my @clean;
	while (@_) {
		if (my $domain = shift @_) {
			$domain =~ s/\.\.//g;
			$domain =~ s/[^\w\.\-]//g;
			push @clean,$domain;
		}
	}
	return @clean
}
